/*
    (c) 2002 - 2022 Juergen Nagel, Jan Hansen
    Northwest German Forest Research Station (https://www.nw-fva.de), 
    Grätzelstr. 2, 37079 Göttingen, Germany
    E-Mail: Jan.Hansen@nw-fva.de
 
    This file is part of the TreeGrOSS libraray.

    TreeGrOSS is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    TreeGrOSS is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with TreeGrOSS. If not, see http://www.gnu.org/licenses/.
*/
package treegross.base;

/**
 * TreeGrOSS : class competition defines the competition indicies
 * http://treegross.sourceforge.net
 *
 * @version 2.0 30-NOV-2004
 * @author	Juergen Nagel
 *
 * For more information see: NAGEL, J. (1999): Konzeptionelle Überlegungen zum
 * schrittweisen Aufbau eines waldwachstumskundlichen Simulationssystems für
 * Nordwestdeutschland. Schriften aus der Forstlichen Fakultät der Universität
 * Göttingen und der Nieders. Forstl. Versuchsanstalt, Band 128, J.D.
 * Sauerländer's Verlag, Frankfurt a.M., S.122
 *
 * or
 *
 * BWINPro User's Manual http://www.nfv.gwdg
 */
public class Competition implements PlugInCompetition {

    //private final static Logger LOGGER = Logger.getLogger(Competition.class.getName());
    
    @Override
    public double getc66(Tree t) {
        double h66 = t.h - (2.0 * (t.h - t.cb) / 3.0);
        double cri = t.cw * 0.5;
        double c66 = t.fac * Math.PI * (cri * cri);       
        for (int i = 0; i < t.st.ntrees; i++) {
            if (t.st.tr[i].out < 0) {
                if (t.st.tr[i].cb >= h66) {
                    cri = t.st.tr[i].cw * 0.5;
                } else {
                    cri = t.st.tr[i].calculateCwAtHeight(h66) * 0.5;
                }
                c66 += t.st.tr[i].fac * Math.PI * cri * cri;
            }
        }
        c66 = c66 / (10000 * t.st.size);
        return c66;
    }

    @Override
    public double getc66c(Tree t) {
        double h66 = t.h - (2.0 * (t.h - t.cb) / 3.0);
        double c66 = 0.0;
        double cri;
        for (int i = 0; i < t.st.ntrees; i++) {
            if (t.st.tr[i].out >= t.st.year && t.st.tr[i].outtype != 1) {
                if (t.st.tr[i].cb >= h66) {
                    cri = t.st.tr[i].cw * 0.5;
                } else {
                    cri = t.st.tr[i].calculateCwAtHeight(h66) * 0.5;
                }
                c66 = c66 + t.st.tr[i].fac * Math.PI * cri * cri;
            }
        }
        c66 = c66 / (10000 * t.st.size);
        //  check if c66c is to high because of measurement errror
        double c66max = getCritcc(t);
        if (c66 > c66max - 0.25) {
            c66 = c66max - t.c66;
        }
        if (c66 < 0.0) {
            c66 = 0.0;
        }
        return c66;
    }
    
    @Override
    public void replaceC66AndC66c(Tree t) {
        // set values to 0
        t.c66 = 0.0; 
        t.c66c = 0.0;        
        
        double h66 = t.h - (2.0 * (t.h - t.cb) / 3.0);
        double cri = t.cw * 0.5;
        double c66 = t.fac * Math.PI * (cri * cri);          
        double c66c = 0.0;
        
        for (int i = 0; i < t.st.ntrees; i++) {
            if (t.st.tr[i].out < 0) {
                if (t.st.tr[i].cb >= h66) {
                    cri = t.st.tr[i].cw * 0.5;
                } else {
                    cri = t.st.tr[i].calculateCwAtHeight(h66) * 0.5;
                }
                c66 += t.st.tr[i].fac * Math.PI * cri * cri;
            } else if (t.st.tr[i].out >= t.st.year && t.st.tr[i].outtype != 1) {
                if (t.st.tr[i].cb >= h66) {
                    cri = t.st.tr[i].cw * 0.5;
                } else {
                    cri = t.st.tr[i].calculateCwAtHeight(h66) * 0.5;
                }
                c66c += t.st.tr[i].fac * Math.PI * cri * cri;
            }
        }
        
        c66 = c66 / (10000 * t.st.size);        
        
        c66c = c66c / (10000 * t.st.size);
        //  check if c66c is to high because of measurement errror
        double c66max = getCritcc(t);
        if (c66c > c66max - 0.25) {
            c66c = c66max - t.c66;
        }
        if (c66c < 0.0) {
            c66c = 0.0;
        }
        
        t.c66 = c66;
        t.c66c = c66c;
    }

    @Override
    public void replaceC66xyAndC66cxy(Tree t, double influenceZoneRadius) {
        // if there are 15 neighbor trees detected the influenece zone needs to be corrected
        // why at all and why 15 ??????
        /*if (t.nNeighbor >= 15 )
         influenceZoneRadius=Math.sqrt(Math.pow(t.st.tr[t.neighbor[t.nNeighbor-1]].x-t.x,2.0)+
         Math.pow(t.st.tr[t.neighbor[t.nNeighbor-1]].y-t.y,2.0));*/

        // influenece zone should be at least 2m radius
        if (influenceZoneRadius < 2.0) {
            influenceZoneRadius = 2.0;
        }

        double h66 = t.cb + (t.h - t.cb) / 3.0; // height for the cross section of subject tree
        t.c66xy = 0.0; // set values to 0
        t.c66cxy = 0.0;

        // 1. we check how many percent of the influence zone is in the
        // stand pologon. The percentage is stored in perc 
        double perc = getPercCircleInStand(influenceZoneRadius, t.x, t.y, t.st);

        // Now we check for each tree             
        // add trees own area
        // this is done in loop over all trees
        //t.c66xy = t.fac * Math.PI * Math.pow((t.cw/2.0),2.0);
        double e;
        /*for (int k=0; k<t.nNeighbor; k++){*/
        //calculate with all trees in stand:
       
        double overlap;
        double percOverlapInStand;
        double cri;
        double distX, distY;
        for (int i = 0; i < t.st.ntrees; i++) {
            //int j = t.neighbor[k];
          
            distX = t.x - t.st.tr[i].x;
            distY = t.y - t.st.tr[i].y;
            e = (distX * distX) + (distY * distY);
            if (e > 0) {
                e = Math.sqrt(e);
            }
            //cri=0.0;
            // neighbours gehen nur max bis influienceZone ein vgl. stand
            // 2. Achtung wenn baum j mit seiner vollen kronenbreite in die influenzsZone ragt,
            //    ist dies nicht für die Kronenbreite in h66 garantiert, dann kann
            //    es sein, dass overlap= 0 ist und getoverlapPerc NaN !!!!
            //    daher: vorher abfragen ob overlap >0
            //    und methode overlap püfen, op überhaupt Punkte im Überlappungsbereich sind
            
            if (e < influenceZoneRadius + t.st.tr[i].cw * 0.5 && t.st.tr[i].h > h66) {                
                if (t.st.tr[i].out < 0) {
                    if (t.st.tr[i].cb >= h66) {
                        cri = t.st.tr[i].cw * 0.5;
                    } else {
                        cri = t.st.tr[i].calculateCwAtHeight(h66) * 0.5;
                    }
                    // reduce overlap area -> use only percentage inside the stand
                    overlap = overlap(cri, influenceZoneRadius, e);
                    //nur wenn Überlappung c66xy erhöhen
                    if (overlap > 0) {
                        percOverlapInStand = getPercOverlapInStand(influenceZoneRadius, t.x, t.y, cri, t.st.tr[i].x, t.st.tr[i].y, t.st);
                        t.c66xy += t.st.tr[i].fac * (overlap * percOverlapInStand);
                    }
                } else if (t.st.tr[i].out >= t.st.year && t.st.tr[i].outtype != 1) {
                    if (t.st.tr[i].cb >= h66) {
                        cri = t.st.tr[i].cw * 0.5;
                    } else {
                        cri = t.st.tr[i].calculateCwAtHeight(h66) * 0.5;
                    }
                    // reduce overlap area -> use only percentage inside the stand
                    overlap = overlap(cri, influenceZoneRadius, e);
                    //nur wenn Überlappung c66xy erhöhen
                    if (overlap > 0) {
                        percOverlapInStand = getPercOverlapInStand(influenceZoneRadius, t.x, t.y, cri, t.st.tr[i].x, t.st.tr[i].y, t.st);
                        t.c66cxy += t.st.tr[i].fac * (overlap * percOverlapInStand);
                    }
                }
            }            
        }
        double div = perc * Math.PI * (influenceZoneRadius * influenceZoneRadius);
        t.c66xy = t.c66xy / div;
        t.c66cxy = t.c66cxy / div;

        //  check if c66c is to high because of measurement errror
        double c66max = getCritcc(t);
        if (t.c66cxy > c66max - 0.25) {
            /*if (t.st != null && t.st.debug) {
                LOGGER.log(Level.INFO, "C66c nicht realistisch: max {0}  t.c66xy={1}  t.c66cxy={2}", new Object[]{c66max, t.c66xy, t.c66cxy});
            }*/
            t.c66cxy = c66max - t.c66xy;
        }
        if (t.c66cxy < 0) {
            t.c66cxy = 0.0;
        }
    }

    /**
     * calculate overlap area of two circle only if they overlap
     */
    private static double overlap(double r1, double r2, double e) {
        double x, y, f, r1s, r2s;
        //f=0.0;
        //r1 should always be the smaller radius
        if (r1 > r2) {
            x = r1;
            r1 = r2;
            r2 = x;
        }
        if (e >= (r1 + r2)) {
            return 0.0;  //no overlap area =0
        }
        r1s = r1 * r1;
        // partly or full overlap
        if ((e + r1) <= r2) {
            return Math.PI * r1s;
        }

        // part. overlap
        x = e * e;
        r2s = r2 * r2;

        y = Math.sqrt((-e + r1 + r2) * (e + r1 - r2) * (e - r1 + r2) * (e + r1 + r2));

        f = (r1s * Math.acos((x + r1s - r2s) / (2 * e * r1)))
                + (r2s * Math.acos((x + r2s - r1s) / (2 * e * r2)))
                - (0.5 * y);

        if (f < 0) {
            return 0;
        }
        if (Double.isNaN(f)) {
            return 0;
        }
        return f;
    }

    /**
     * get percentage of influence zone area inside the stand
     *
     * @param radius
     * @param x
     * @param y
     * @param st
     * @return
     */
    public double getPercCircleInStand(double radius, double x, double y, Stand st) {
        int pan = 0; // number of points inside influence zone total
        int pani = 0; //number of points inside influence zone and inside plot area        
        double distXS, distYS;
        double xpx = x - radius + radius * 0.05; // x - radius + radius / 20;
        double ypy;
        int in;
        double radiusS = radius * radius;
        for (int k = 0; k < 10; k++) {
            ypy = y - radius + radius * 0.05; // y - radius + radius / 20;
            distXS = (xpx - x) * (xpx - x);
            for (int kk = 0; kk < 10; kk++) {
                distYS = (ypy - y) * (ypy - y);
                //if (Math.sqrt(distXS + distYS) <= radius) {
                if ((distXS + distYS) <= radiusS) {
                    pan++;
                    in = pnpoly(xpx, ypy, st);
                    if (in != 0) {
                        pani++;
                    }
                }
                ypy += 0.2 * radius; // 2.0 * radius / 10.0;
            }
            xpx += 0.2 * radius; // 2.0 * radius / 10.0;
        }
        return (double) (pani) / (double) (pan);
    }

    /**
     * get percentage of overlap area inside the stand
     *
     * @param radius_cz: radius of influenze zone
     * @param x_cz: x coordinate of influenze zone (the tree with these
     * influenze zone)
     * @param y_cz: y coordinate of influenze zone (the tree with these
     * influenze zone)
     * @param radius_crown: the crown radius in h66 or max crone radius (cri) of
     * a neighbour tree
     * @param x_crown: the x coordinate of the neighbour tree;
     * @param y_crown: the y coordinate of the neighbour tree;
     * @param st: the stand containing both trees
     *
     * @return returns the percentage of overlap area in stand polygon
     */
    private double getPercOverlapInStand(double radius_cz, double x_cz, double y_cz,
            double radius_crown, double x_crown, double y_crown, Stand st) {
        int pan = 0; // number of points inside overlap zone total
        int pani = 0; //number of points inside overlap zone and inside plot area
        double xpx = x_crown - radius_crown + radius_crown * 0.05; // x_crown - radius_crown + radius_crown / 20;
        double ypy;
        int in;
        double dXC, dYC, dXI, dYI;
        double radiusCrownS = radius_crown * radius_crown;
        double radiusCzS = radius_cz * radius_cz;
        for (int k = 0; k < 10; k++) {
            ypy = y_crown - radius_crown + radius_crown * 0.05; // y_crown - radius_crown + radius_crown / 20;
            dXC = (xpx - x_crown) * (xpx - x_crown);
            dXI = (xpx - x_cz) * (xpx - x_cz);
            for (int kk = 0; kk < 10; kk++) {
                dYC = (ypy - y_crown) * (ypy - y_crown);
                // is point in crown circle
                //if (Math.sqrt(dXC + dYC) <= radius_crown) {
                if ((dXC + dYC) <= radiusCrownS) {    
                    // ist point in influencezone circle
                    dYI = (ypy - y_cz) * (ypy - y_cz);
                    //if (Math.sqrt(dXI + dYI) <= radius_cz) {
                    if ((dXI + dYI) <= radiusCzS) {
                        pan++;
                        in = pnpoly(xpx, ypy, st);
                        if (in != 0) {
                            pani++;
                        }
                    }
                }
                ypy += 0.2 * radius_crown; // -> 2.0 * radius_crown * 0.1; // 2.0 * radius_crown / 10.0;
            }
            xpx += 0.2 * radius_crown;  // -> 2.0 * radius_crown * 0.1; // 2.0 * radius_crown / 10.0;
        }
        // neu zur Sicherheit
        if (pan > 0) {
            return (double) (pani) / (double) (pan);
        } else {
            return 0;
        }
    }
    
    /**
     * check if a point is in polygon , if return is 0 then outside
     *
     * @param x
     * @param y
     * @param st
     * @return
     */
    public int pnpoly(double x, double y, Stand st) {
        int i, j, c, m;
        c = 0;
        m = st.ncpnt;
        //System.out.println("pnpoly "+m+" "+x+" y "+y);
        j = m - 1;
        for (i = 0; i < m; i++) {
            if ((((st.cpnt[i].y <= y) && (y < st.cpnt[j].y))
                    || ((st.cpnt[j].y <= y) && (y < st.cpnt[i].y)))
                    && (x < (st.cpnt[j].x - st.cpnt[i].x) * (y - st.cpnt[i].y)
                    / (st.cpnt[j].y - st.cpnt[i].y) + st.cpnt[i].x)) {
                if (c == 0) {
                    c = 1;
                } else {
                    c = 0;
                }
            }
            j = i;
        }
        return c;
    }

    /**
     * get critical crown closure, this is needed to check c66c-Values
     *
     */
    private double getCritcc(Tree t) {
        double critcc;
        Tree atree = new Tree();
        atree.sp = t.sp;
        atree.st = t.st;
        atree.code = t.code;
        atree.d = t.d;
        atree.h = t.h;
        atree.age = t.age;
        atree.cb = atree.calculateCb();
        // hier wird cw berechnet
        atree.cw = atree.calculateCw();
        double maxBa = atree.calculateMaxBasalArea();
        double dS = (atree.d * 0.005) * (atree.d * 0.005);
        double cwS = (atree.cw * 0.5) * (atree.cw * 0.5);
        double maxNha = maxBa / (Math.PI * dS);
        critcc = (maxNha * Math.PI * cwS) * 0.0001; // / 10000.0;
        return critcc;
    }
}
