/*
    (c) 2002 - 2022 Juergen Nagel, Jan Hansen
    Northwest German Forest Research Station (https://www.nw-fva.de), 
    Grätzelstr. 2, 37079 Göttingen, Germany
    E-Mail: Jan.Hansen@nw-fva.de
 
    This file is part of the TreeGrOSS libraray.

    TreeGrOSS is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    TreeGrOSS is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with TreeGrOSS. If not, see http://www.gnu.org/licenses/.
 */
package treegross.base;

import treegross.random.RandomNumber;

/**
 *
 * @author jhansen
 */
public class ScaleManager {

    public final static int SCALE_AUTO = 0;
    public final static int SCALE_FIXED = 1;
    public final static int SCALE_NOT = 2;

    private int nThreads;
    private int nt2;
    private int scaleMethod;

    /*
     autoThreshold: Min. number of trees to force usage of parallel competition updating.
     Used in SCALE_AUTO mode only.
     */
    private int autoThreshold = 500;

    /**
     * Creates a new ScaleManager
     *
     * @param scaleMethod the method to use
     * @param scale the number of Threads to start parallel
     * @param autoTheshold Min. number of trees to force usage of parallel
     * competition updating in SCALE_AUTO
     */
    public ScaleManager(int scaleMethod, int scale, int autoTheshold) {
        setAutoTheshold(autoTheshold);
        setScaleMethod(scaleMethod);
        setScale(scale);
    }

    public ScaleManager() {
        this(SCALE_AUTO, Runtime.getRuntime().availableProcessors(), 500);
    }

    public final void setScale(int scale) {
        nThreads = scale;
        if (nThreads < 1 || nThreads > 32) {
            nThreads = 1;
        }
        nt2 = nThreads * 3;
    }

    public final void setAutoTheshold(int t) {
        if (t >= 0) {
            autoThreshold = t;
        } else {
            autoThreshold = 0;
        }
    }

    public final void setScaleMethod(int m) {
        this.scaleMethod = m;
        if (this.scaleMethod < 0 || this.scaleMethod > 2) {
            this.scaleMethod = SCALE_NOT;
        }
    }

    public int getScaleMethod() {
        return scaleMethod;
    }

    public void updateCompetition(Stand st) {
        int localScaleMethod = scaleMethod;
        //long msS = System.currentTimeMillis();
        if (nThreads == 1 || st.nTreesAlive < nt2) {
            localScaleMethod = SCALE_NOT;
        }
        //System.out.println("updating mechanism: "+scaleMethod+" "+nThreads);
        switch (localScaleMethod) {
            case SCALE_AUTO:
                if (st.nTreesAlive >= autoThreshold) {
                    updateCompetitionFixScale(st);
                } else {
                    updateCompetitionNoScale(st);
                }
                break;
            case SCALE_FIXED:
                updateCompetitionFixScale(st);
                break;
            case SCALE_NOT:
                updateCompetitionNoScale(st);
                break;
        }
        //System.out.println(st.ntrees+":"+(System.currentTimeMillis()-msS));
    }

    public void updateCompetitionMortality(Stand st, int numberOfCandidates, int[] treeNo) {
        //long msS = System.currentTimeMillis();
        if (numberOfCandidates <= 0) {
            return;
        }
        int localScaleMethod = scaleMethod;
        if (nThreads == 1 || numberOfCandidates < nt2) {
            localScaleMethod = SCALE_NOT;
        }
        //System.out.println("updating mechanism: "+scaleMethod+" "+nThreads);
        //System.out.println("ucm "+numberOfCandidates+" "+localScaleMethod);
        switch (localScaleMethod) {
            case SCALE_AUTO:
                if (st.nTreesAlive >= autoThreshold) {
                    updateCompetitionMortalityFixScale(st, numberOfCandidates, treeNo);
                } else {
                    updateCompetitionMortalityNoScale(st, numberOfCandidates, treeNo);
                }
                break;
            case SCALE_FIXED:
                updateCompetitionMortalityFixScale(st, numberOfCandidates, treeNo);
                break;
            case SCALE_NOT:
                updateCompetitionMortalityNoScale(st, numberOfCandidates, treeNo);
                break;
        }
        //System.out.println(st.ntrees+":"+(System.currentTimeMillis()-msS));
    }

    public void updateCrown(Stand st) {
        //long msS = System.currentTimeMillis();
        int localScaleMethod = scaleMethod;
        if (nThreads == 1 || st.nTreesAlive < nt2) {
            localScaleMethod = SCALE_NOT;
        }
        //System.out.println("updating mechanism: "+scaleMethod+" "+nThreads);
        switch (localScaleMethod) {
            case SCALE_AUTO:
                if (st.nTreesAlive >= autoThreshold) {
                    updateCrownFixScale(st);
                } else {
                    updateCrownNoScale(st);
                }
                break;
            case SCALE_FIXED:
                updateCrownFixScale(st);
                break;
            case SCALE_NOT:
                updateCrownNoScale(st);
                break;
        }
        //System.out.println(st.ntrees+":"+(System.currentTimeMillis()-msS));
    }

    void growTrees(Stand st, int period, RandomNumber random) {
        //long msS = System.currentTimeMillis();
        int localScaleMethod = scaleMethod;
        if (nThreads == 1 || st.nTreesAlive < nt2) {
            localScaleMethod = SCALE_NOT;
        }
        //System.out.println("updating mechanism: "+scaleMethod+" "+nThreads);
        switch (localScaleMethod) {
            case SCALE_AUTO:
                if (st.nTreesAlive >= autoThreshold) {
                    growFixScale(st, period, random);
                } else {
                    growNoScale(st, period, random);
                }
                break;
            case SCALE_FIXED:
                growFixScale(st, period, random);
                break;
            case SCALE_NOT:
                growNoScale(st, period, random);
                break;
        }
        //System.out.println(st.ntrees+":"+(System.currentTimeMillis()-msS));
    }

    private void updateCompetitionMortalityNoScale(Stand st, int numberOfCandidates, int[] treeNo) {
        for (int i = 0; i < numberOfCandidates; i++) {
            st.tr[treeNo[i]].updateCompetition();
        }
    }

    private void updateCompetitionMortalityFixScale(Stand st, int numberOfCandidates, int[] treeNo) {
        UpdateThreadCompetitionMortality[] ut = new UpdateThreadCompetitionMortality[nThreads];
        int avgSize = numberOfCandidates / nThreads;
        int rest = numberOfCandidates % nThreads;
        int s, e;
        e = rest + avgSize;
        s = 0;
        for (int i = 0; i < ut.length; i++) {
            //System.out.println("thread "+i+": "+st.ntrees+" se:"+s+" / "+e);
            ut[i] = new UpdateThreadCompetitionMortality(st, s, e, treeNo);
            ut[i].start();
            s = e;
            e += avgSize;
        }
        for (UpdateThreadCompetitionMortality ut1 : ut) {
            try {
                ut1.join();
            } catch (InterruptedException ex) {
            }
        }
    }

    private void updateCompetitionNoScale(Stand st) {
        for (int i = 0; i < st.ntrees && !st.stop; i++) {
            if (st.tr[i].out < 0) {
                st.tr[i].updateCompetition();
            }
        }
    }

    private void updateCompetitionFixScale(Stand st) {
        UpdateThreadCompetition[] ut = new UpdateThreadCompetition[nThreads];
        int avgSize = st.ntrees / nThreads;
        int rest = st.ntrees % nThreads;
        int s, e;
        e = rest + avgSize;
        s = 0;
        for (int i = 0; i < ut.length; i++) {
            //System.out.println("thread "+i+": "+st.ntrees+" se:"+s+" / "+e);
            ut[i] = new UpdateThreadCompetition(st, s, e);
            ut[i].start();
            s = e;
            e += avgSize;
        }
        for (UpdateThreadCompetition ut1 : ut) {
            try {
                ut1.join();
            } catch (InterruptedException ex) {
            }
        }
    }

    private void updateCrownNoScale(Stand st) {
        for (int i = 0; i < st.ntrees && !st.stop; i++) {
            if (st.tr[i].out < 0) {
                st.tr[i].updateCrown();
            }
        }
    }

    private void updateCrownFixScale(Stand st) {
        UpdateThreadCrown[] ut = new UpdateThreadCrown[nThreads];
        int avgSize = st.ntrees / nThreads;
        int rest = st.ntrees % nThreads;
        int s, e;
        e = rest + avgSize;
        s = 0;
        for (int i = 0; i < ut.length; i++) {
            //System.out.println("thread "+i+": "+st.ntrees+" se:"+s+" / "+e);
            ut[i] = new UpdateThreadCrown(st, s, e);
            ut[i].start();
            s = e;
            e += avgSize;
        }
        for (UpdateThreadCrown ut1 : ut) {
            try {
                ut1.join();
            } catch (InterruptedException ex) {
            }
        }
    }

    private void growNoScale(Stand st, int period, RandomNumber random) {
        for (int i = 0; i < st.ntrees && !st.stop; i++) {
            if (st.tr[i].out < 0) {
                st.tr[i].grow(period, random);
            }
        }
    }

    private void growFixScale(Stand st, int period, RandomNumber random) {
        GrowThread[] ut = new GrowThread[nThreads];
        int avgSize = st.ntrees / nThreads;
        int rest = st.ntrees % nThreads;
        int s, e;
        e = rest + avgSize;
        s = 0;
        for (int i = 0; i < ut.length; i++) {
            //System.out.println("thread "+i+": "+st.ntrees+" se:"+s+" / "+e);
            ut[i] = new GrowThread(st, s, e, period, random);
            ut[i].start();
            s = e;
            e += avgSize;
        }
        for (GrowThread ut1 : ut) {
            try {
                ut1.join();
            } catch (InterruptedException ex) {
            }
        }
    }

    private class UpdateThreadCompetition extends Thread {

        private final Stand st;
        private final int s, e;

        public UpdateThreadCompetition(Stand st, int startIndex, int endIndex) {
            this.st = st;
            s = startIndex;
            e = endIndex;
        }

        @Override
        public void run() {
            for (int i = s; i < e && !st.stop; i++) {
                if (st.tr[i].out < 0) {
                    st.tr[i].updateCompetition();
                }
            }
        }
    }

    private class UpdateThreadCompetitionMortality extends Thread {

        private final Stand st;
        private final int s, e;
        private final int[] treeNo;

        public UpdateThreadCompetitionMortality(Stand st, int startIndex, int endIndex, int[] treeNo) {
            this.st = st;
            s = startIndex;
            e = endIndex;
            this.treeNo = treeNo;
        }

        @Override
        public void run() {
            for (int i = s; i < e && !st.stop; i++) {
                st.tr[treeNo[i]].updateCompetition();
            }
        }
    }

    private class UpdateThreadCrown extends Thread {

        private final Stand st;
        private final int s, e;

        public UpdateThreadCrown(Stand st, int startIndex, int endIndex) {
            this.st = st;
            s = startIndex;
            e = endIndex;
        }

        @Override
        public void run() {
            for (int i = s; i < e && !st.stop; i++) {
                if (st.tr[i].out < 0) {
                    st.tr[i].updateCrown();
                }
            }
        }
    }

    private class GrowThread extends Thread {

        private final Stand st;
        private final int s, e;
        private final int period;
        private final RandomNumber random;

        public GrowThread(Stand st, int startIndex, int endIndex, int period, RandomNumber random) {
            this.st = st;
            s = startIndex;
            e = endIndex;
            this.period = period;
            this.random = random.clone();
        }

        @Override
        public void run() {
            for (int i = s; i < e && !st.stop; i++) {
                if (st.tr[i].out < 0) {
                    st.tr[i].grow(period, random);
                }
            }
        }
    }
}
