/*
    (c) 2002 - 2022 Juergen Nagel, Jan Hansen
    Northwest German Forest Research Station (https://www.nw-fva.de), 
    Grätzelstr. 2, 37079 Göttingen, Germany
    E-Mail: Jan.Hansen@nw-fva.de
 
    This file is part of the TreeGrOSS libraray.

    TreeGrOSS is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    TreeGrOSS is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with TreeGrOSS. If not, see http://www.gnu.org/licenses/.
*/

package treegross.base;

import treegross.functions.TGTextFunction;
import treegross.functions.TGClassFunction;
import treegross.functions.TGFunction;
import java.util.HashMap;
import java.util.Iterator;
import java.io.*;
import java.net.*;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 *
 * @author jhansen
 */
public class SpeciesDefMap {

    private HashMap<Integer, SpeciesDef> spcdef;
    private URL actualurl;
    private boolean loaded = false;
    //private final String stdModel = "ForestSimulatorNWGermanyBC4_1";
    private final String stdModel = "ForestSimulatorNWGermany6";

    private final static Logger LOGGER = Logger.getLogger(SpeciesDefMap.class.getName());

    public SpeciesDefMap() {
        spcdef = null;
        loaded = false;
        actualurl = null;
    }

    public String getDefaultModelName(){
        return stdModel;
    }
    
    public void reload() {
        if (loaded && actualurl != null) {
            readFromURL(actualurl);
        }
    }

    public void readInternal(String name) {
        URL url;
        if (name != null) {
            url = getClass().getResource("/treegross/model/" + name + ".xml");
        } else {
            url = getClass().getResource("/treegross/model/" + stdModel + ".xml");
        }
        readFromURL(url);
    }

    public void readFromPath(String path) {
        try {
            URL url = new File(path).toURI().toURL();
            readFromURL(url);
        } catch (MalformedURLException e) {
            LOGGER.log(Level.SEVERE, "reading xml file: ", e);
        }
    }

    public void readFromURL(URL url) {
        try {
            readXML(url);
            actualurl = url;
        } catch (IOException e) {
            LOGGER.log(Level.SEVERE, "reading xml file: ", e);
        }
    }

    private void readXML(URL url) throws IOException {
        URLConnection urlcon = url.openConnection();
        actualurl = url;
        readXMLStream(urlcon.getInputStream());
    }    

    /**
     * read a xml file with SpeciesDefinition tags
     * 
     * changes
     * -------------------------------------------------------------------------
     * jhansen, 02.12.2015:
     * read xml without 3rd party libraray (jdom)
     * recursive inheritance over several (not only one) levels is now possible
     * -------------------------------------------------------------------------
     * 
     * @param imps the input stream to read xml data from
     */
    public void readXMLStream(InputStream imps) {
        loaded = false;
        spcdef = new HashMap<>();
        try {
            Document d = DocumentBuilderFactory.newInstance().newDocumentBuilder().parse(imps);
            NodeList sDefs = d.getElementsByTagName("SpeciesDefinition");            
            NodeList modelNodes;
            int length = sDefs.getLength();
            //System.out.println("length: " + length);
            Node n, m;
            String tc, nn;
            int code;
            for (int i = 0; i < length; i++) {
                n = sDefs.item(i);                
                //System.out.println("next :" + n.getNodeName());            
                modelNodes = n.getChildNodes();
                for (int j = 0; j < modelNodes.getLength(); j++) {
                    m = modelNodes.item(j);
                    if (m.getNodeType() == Node.ELEMENT_NODE) {
                        tc = m.getTextContent();
                        nn = m.getNodeName().toLowerCase();                        
                        //System.out.println("\t" + nn + ": '" + tc + "'");
                        if (nn.equals("code") && tc != null && tc.length() > 0) {
                            code = Integer.parseInt(tc.trim());
                            addSpeciesDef(code, sDefs);
                            break;
                        }
                    }
                }
            }
            loaded = true;
        } catch (SAXException | IOException | ParserConfigurationException | NumberFormatException ex) {
            LOGGER.log(Level.SEVERE, "Parsing species definition failed!", ex);
        } finally {
            if (imps != null) {
                try {
                    imps.close();
                } catch (IOException ex) {
                    LOGGER.log(Level.SEVERE, "treegross", ex);
                }
            }
        }
    }

    private void addSpeciesDef(int code, NodeList specDefNodes) {
        if (spcdef.containsKey(code)) {
            //System.out.println(code + " already defined returning...");
            return;
        }
        //System.out.println("adding " + code + "...");
        NodeList modelNodes;
        Node n, m;
        String tc, nn;
        int nCode, hlc;
        boolean found = false;
        for (int i = 0; i < specDefNodes.getLength(); i++) {
            n = specDefNodes.item(i);
            modelNodes = n.getChildNodes();
            for (int j = 0; j < modelNodes.getLength(); j++) {
                m = modelNodes.item(j);
                if (m.getNodeType() == Node.ELEMENT_NODE && "code".equalsIgnoreCase(m.getNodeName())) {
                    nCode = Integer.parseInt(m.getTextContent());
                    if (code == nCode) {
                        found = true;
                        break;
                    }
                }
            }
            if (found) {
                SpeciesDef sd = new SpeciesDef();
                sd.code = code;
                hlc = -1;
                for (int j = 0; j < modelNodes.getLength(); j++) {
                    m = modelNodes.item(j);
                    if (m.getNodeType() == Node.ELEMENT_NODE) {
                        //tc = m.getChildNodes().item(0).getNodeValue();
                        tc = m.getTextContent();
                        nn = m.getNodeName().toLowerCase();
                        //System.out.println("\t" + nn + ": '" + tc + "'");
                        if (tc != null && tc.length() > 0) {
                            tc = tc.trim();
                            switch (nn) {
                                case "handledlikecode":
                                    hlc = Integer.parseInt(tc);
                                    sd.handledLikeCode = hlc;                                    
                                    break;
                                case "shortname":
                                    sd.shortName = tc;                                    
                                    break;
                                case "longname":
                                    sd.longName = tc;                                    
                                    break;
                                case "latinname":
                                    sd.latinName = tc;                                    
                                    break;
                                case "internalcode":
                                    sd.internalCode = Integer.parseInt(tc);                                    
                                    break;
                                case "codegroup":
                                    sd.codeGroup = Integer.parseInt(tc);                                    
                                    break;
                                case "heightcurve":
                                    sd.heightCurve = Integer.parseInt(tc);                                    
                                    break;
                                case "crowntype":
                                    sd.crownType = Integer.parseInt(tc);                                    
                                    break;
                                case "heightincrementerror":
                                    sd.heightIncrementError = Double.parseDouble(tc);
                                    break;
                                case "diameterincrementerror":
                                    sd.diameterIncrementError = Double.parseDouble(tc);                                    
                                    break;
                                case "maximumage":
                                    sd.maximumAge = Integer.parseInt(tc);                                    
                                    break;
                                case "ingrowth":
                                    sd.ingrowthXML = tc;                                    
                                    break;
                                case "targetdiameter":
                                    sd.targetDiameter = Double.parseDouble(tc);                                    
                                    break;
                                case "croptreenumber":
                                    sd.cropTreeNumber = stripCommentsFromInt(tc, -9);                                    
                                    break;
                                case "heightofthinningstart":
                                    sd.heightOfThinningStart = Double.parseDouble(tc);                                    
                                    break;
                                case "moderatethinning":
                                    sd.moderateThinning = tc;                                    
                                    break;
                                case "color":
                                    sd.colorXML = tc.replace(",", ";");                                    
                                    break;
                                case "competition":
                                    sd.competitionXML = tc;                                    
                                    break;
                                case "mortality":
                                    sd.mortalityXML = tc;                                    
                                    break;
                                case "taperfunction":
                                    sd.taperFunctionXML = tc;                                    
                                    break;
                                case "stemvolumefunction":
                                    sd.stemVolumeFunctionXML = tc;                                    
                                    break;
                                case "coarserootbiomass":
                                    sd.coarseRootBiomass = tc;                                    
                                    break;
                                case "smallrootbiomass":
                                    sd.smallRootBiomass = tc;                                    
                                    break;
                                case "finerootbiomass":
                                    sd.fineRootBiomass = tc;                                    
                                    break;
                                case "totalrootbiomass":
                                    sd.totalRootBiomass = tc;                                    
                                    break;
                                //TGFunctions
                                case "uniformheightcurvexml":
                                    sd.uniformHeightCurveXML = initTGFunction(tc);                                    
                                    break;
                                case "heightvariation":
                                    sd.heightVariationXML = initTGFunction(tc);                                    
                                    break;
                                case "diameterdistributionxml":
                                    sd.diameterDistributionXML = initTGFunction(tc);                                    
                                    break;
                                case "volumefunctionxml":
                                    sd.volumeFunctionXML = initTGFunction(tc);                                    
                                    break;
                                case "crownwidth":
                                    sd.crownwidthXML = initTGFunction(tc);                                    
                                    break;
                                case "crownbase":
                                    sd.crownbaseXML = initTGFunction(tc);                                    
                                    break;
                                case "siteindex":
                                    sd.siteindexXML = initTGFunction(tc);                                    
                                    break;
                                case "siteindexheight":
                                    sd.siteindexHeightXML = initTGFunction(tc);                                    
                                    break;
                                case "potentialheightincrement":
                                    sd.potentialHeightIncrementXML = initTGFunction(tc);                                    
                                    break;
                                case "heightincrement":
                                    sd.heightIncrementXML = initTGFunction(tc);                                    
                                    break;
                                case "diameterincrement":
                                    sd.diameterIncrementXML = initTGFunction(tc);                                    
                                    break;
                                case "maximumdensity":
                                    sd.maximumDensityXML = initTGFunction(tc);                                    
                                    break;
                                case "decay":
                                    sd.decayXML = initTGFunction(tc);                                    
                                    break;
                                case "diametertreeerror":
                                    sd.diameterTreeErrorXML = tc;                                    
                                    break;
                            }
                        }
                    }
                }               
                if (hlc >= 0 && hlc != sd.code) {
                    //System.out.println("\tdefine from parent: " + hlc);
                    if (!spcdef.containsKey(hlc)) {
                        // read recusive
                        //System.out.println("\t\trecursion");
                        addSpeciesDef(hlc, specDefNodes);
                    }
                    fillUpWithParentValues(sd, spcdef.get(hlc));
                }
                String[] rgb = sd.colorXML.split(";");
                sd.colorRed = Integer.parseInt(rgb[0]);
                sd.colorGreen = Integer.parseInt(rgb[1]);
                sd.colorBlue = Integer.parseInt(rgb[2]);
                
                sd.initPlugins();
                
                sd.setDefined(true);
                spcdef.put(sd.code, sd);
                break;
            }
        }
    }

    private void fillUpWithParentValues(SpeciesDef child, SpeciesDef parent) {
        if (!isFunctionDefined(child.uniformHeightCurveXML)) {
            child.uniformHeightCurveXML = saveClone(parent.uniformHeightCurveXML);
        }
        if (!isFunctionDefined(child.heightVariationXML)) {
            child.heightVariationXML = saveClone(parent.heightVariationXML);
        }
        if (!isFunctionDefined(child.diameterDistributionXML)) {
            child.diameterDistributionXML = saveClone(parent.diameterDistributionXML);
        }
        if (!isFunctionDefined(child.volumeFunctionXML)) {
            child.volumeFunctionXML = saveClone(parent.volumeFunctionXML);
        }
        if (!isFunctionDefined(child.crownwidthXML)) {
            child.crownwidthXML = saveClone(parent.crownwidthXML);
        }
        if (!isFunctionDefined(child.crownbaseXML)) {
            child.crownbaseXML = saveClone(parent.crownbaseXML);
        }
        if (!isFunctionDefined(child.siteindexXML)) {
            child.siteindexXML = saveClone(parent.siteindexXML);
        }
        if (!isFunctionDefined(child.siteindexHeightXML)) {
            child.siteindexHeightXML = saveClone(parent.siteindexHeightXML);
        }
        if (!isFunctionDefined(child.potentialHeightIncrementXML)) {
            child.potentialHeightIncrementXML = saveClone(parent.potentialHeightIncrementXML);
        }
        if (!isFunctionDefined(child.heightIncrementXML)) {
            child.heightIncrementXML = saveClone(parent.heightIncrementXML);
        }
        if (!isFunctionDefined(child.diameterIncrementXML)) {
            child.diameterIncrementXML = saveClone(parent.diameterIncrementXML);
        }
        if (!isFunctionDefined(child.maximumDensityXML)) {
            child.maximumDensityXML = saveClone(parent.maximumDensityXML);
        }
        if (!isFunctionDefined(child.decayXML)) {
            child.decayXML = saveClone(parent.decayXML);
        }

        if (child.crownType < 0) {
            child.crownType = parent.crownType;
        }
        if (child.heightIncrementError < 0) {
            child.heightIncrementError = parent.heightIncrementError;
        }
        if (child.diameterIncrementError < 0) {
            child.diameterIncrementError = parent.diameterIncrementError;
        }
        if (child.maximumAge < 0) {
            child.maximumAge = parent.maximumAge;
        }
        if (child.ingrowthXML.isEmpty()) {
            child.ingrowthXML = parent.ingrowthXML;
        }
        if (child.heightCurve < 0) {
            child.heightCurve = parent.heightCurve;
        }
        if (child.targetDiameter < 0) {
            child.targetDiameter = parent.targetDiameter;
        }
        if (child.cropTreeNumber < 0) {
            child.cropTreeNumber = parent.cropTreeNumber;
        }
        if (child.heightOfThinningStart < 0) {
            child.heightOfThinningStart = parent.heightOfThinningStart;
        }
        if (child.moderateThinning.isEmpty()) {
            child.moderateThinning = parent.moderateThinning;
        }
        if (child.colorXML.isEmpty()) {
            child.colorXML = parent.colorXML;
        }
        if (child.competitionXML.isEmpty()) {
            child.competitionXML = parent.competitionXML;
        }
        if (child.mortalityXML.isEmpty()) {
            child.mortalityXML = parent.mortalityXML;
        }
        if (child.taperFunctionXML.isEmpty()) {
            child.taperFunctionXML = parent.taperFunctionXML;
        }
        if (child.stemVolumeFunctionXML.isEmpty()) {
            try {
                child.stemVolumeFunctionXML = parent.stemVolumeFunctionXML;
            } catch (Exception e) {
                LOGGER.log(Level.INFO, "Schaftholz ist: {0}", child.stemVolumeFunctionXML);
            }
        }
        if (child.coarseRootBiomass.isEmpty()) {
            child.coarseRootBiomass = parent.coarseRootBiomass;
        }
        if (child.smallRootBiomass.isEmpty()) {
            child.smallRootBiomass = parent.smallRootBiomass;
        }
        if (child.fineRootBiomass.isEmpty()) {
            child.fineRootBiomass = parent.fineRootBiomass;
        }
        if (child.totalRootBiomass.isEmpty()) {
            child.totalRootBiomass = parent.totalRootBiomass;
        }
    }

    private boolean isFunctionDefined(TGFunction function) {
        return (function != null && function.toString().length() > 0);
    }

    private TGFunction saveClone(TGFunction parentFunction) {
        if (parentFunction == null) {
            return null;
        }
        return parentFunction.clone();
    }

    /*
    
    public void readXMLStream(InputStream imps) throws IOException, org.jdom.JDOMException {
        spcdef = new HashMap<>();
        SAXBuilder builder = new SAXBuilder();
        Document doc = builder.build(imps);
        DocType docType = doc.getDocType();
        Element rm = doc.getRootElement();
        List list = rm.getChildren("SpeciesDefinition");
        Iterator i = list.iterator();
        Element def;
        int code, m, handledLikeCode;

        while (i.hasNext()) {
            def = (Element) i.next();
            code = Integer.parseInt(def.getChild("Code").getText());
            handledLikeCode = Integer.parseInt(def.getChild("HandledLikeCode").getText());
            spcdef.put(code, new SpeciesDef());
            SpeciesDef actual = spcdef.get(code);
            define(code, actual, def, handledLikeCode);
            if (code != handledLikeCode) {
                boolean found = false;
                Iterator j = list.iterator();
                while (j.hasNext() && !found) {
                    Element parent_def = (Element) j.next();
                    int code_parent = Integer.parseInt(parent_def.getChild("Code").getText());
                    if (handledLikeCode == code_parent) {
                        overload(actual, parent_def);
                        found = true;
                    }
                }
            }
            m = actual.colorXML.indexOf(";");
            actual.colorRed = Integer.parseInt(actual.colorXML.substring(0, m));
            actual.colorXML = actual.colorXML.substring(m + 1);
            m = actual.colorXML.indexOf(";");
            actual.colorGreen = Integer.parseInt(actual.colorXML.substring(0, m));
            actual.colorXML = actual.colorXML.substring(m + 1);
            actual.colorBlue = Integer.parseInt(actual.colorXML);
            actual.setDefined(true);
        }
        loaded = true;
    }

    private void overload(SpeciesDef actual, Element with) {
        if (actual.uniformHeightCurveXML.toString().length() == 0) {
            actual.uniformHeightCurveXML = initTGFunction(with.getChild("UniformHeightCurveXML").getText());
        }
        if (actual.heightVariationXML.toString().length() == 0) {
            actual.heightVariationXML = initTGFunction(with.getChild("HeightVariation").getText());
        }
        if (actual.diameterDistributionXML.toString().length() == 0) {
            actual.diameterDistributionXML = initTGFunction(with.getChild("DiameterDistributionXML").getText());
        }
        if (actual.volumeFunctionXML.toString().length() == 0) {
            actual.volumeFunctionXML = initTGFunction(with.getChild("VolumeFunctionXML").getText());
        }
        if (actual.crownwidthXML.toString().length() == 0) {
            actual.crownwidthXML = initTGFunction(with.getChild("Crownwidth").getText());
        }
        if (actual.crownbaseXML.toString().length() == 0) {
            actual.crownbaseXML = initTGFunction(with.getChild("Crownbase").getText());
        }
        if (actual.siteindexXML.toString().length() == 0) {
            actual.siteindexXML = initTGFunction(with.getChild("SiteIndex").getText());
        }
        if (actual.siteindexHeightXML.toString().length() == 0) {
            actual.siteindexHeightXML = initTGFunction(with.getChild("SiteIndexHeight").getText());
        }
        if (actual.potentialHeightIncrementXML.toString().length() == 0) {
            actual.potentialHeightIncrementXML = initTGFunction(with.getChild("PotentialHeightIncrement").getText());
        }
        if (actual.heightIncrementXML.toString().length() == 0) {
            actual.heightIncrementXML = initTGFunction(with.getChild("HeightIncrement").getText());
        }
        if (actual.diameterIncrementXML.toString().length() == 0) {
            actual.diameterIncrementXML = initTGFunction(with.getChild("DiameterIncrement").getText());
        }
        if (actual.maximumDensityXML.toString().length() == 0) {
            actual.maximumDensityXML = initTGFunction(with.getChild("MaximumDensity").getText());
        }
        if (actual.decayXML.toString().length() == 0) {
            actual.decayXML = initTGFunction(with.getChild("Decay").getText());
        }

        if (actual.crownType < 0) {
            actual.crownType = Integer.parseInt(with.getChild("CrownType").getText());
        }
        if (actual.heightIncrementError < 0) {
            actual.heightIncrementError = Double.parseDouble(with.getChild("HeightIncrementError").getText());
        }
        if (actual.diameterIncrementError < 0) {
            actual.diameterIncrementError = Double.parseDouble(with.getChild("DiameterIncrementError").getText());
        }
        if (actual.maximumAge < 0) {
            actual.maximumAge = Integer.parseInt(with.getChild("MaximumAge").getText());
        }
        if (actual.ingrowthXML.trim().length() < 1) {
            actual.ingrowthXML = with.getChild("Ingrowth").getText();
        }
        if (actual.heightCurve < 0) {
            actual.heightCurve = Integer.parseInt(with.getChild("HeightCurve").getText());
        }
        if (actual.targetDiameter < 0) {
            actual.targetDiameter = Double.parseDouble(with.getChild("TargetDiameter").getText());
        }
        if (actual.cropTreeNumber < 0) {
            actual.cropTreeNumber = stripCommentsFromInt(with.getChild("CropTreeNumber").getText(), 100);
        }
        if (actual.heightOfThinningStart < 0) {
            actual.heightOfThinningStart = Double.parseDouble(with.getChild("HeightOfThinningStart").getText());
        }
        if (actual.moderateThinning.trim().length() < 1) {
            actual.moderateThinning = with.getChild("ModerateThinning").getText();
        }
        if (actual.colorXML.trim().length() < 1) {
            actual.colorXML = with.getChild("Color").getText();
        }
        if (actual.competitionXML.trim().length() < 1) {
            actual.competitionXML = with.getChild("Competition").getText();
        }
        if (actual.taperFunctionXML.trim().length() < 1) {
            actual.taperFunctionXML = with.getChild("TaperFunction").getText();
        }
        if (actual.stemVolumeFunctionXML.trim().length() < 1) {
            try {
                actual.stemVolumeFunctionXML = with.getChild("StemVolumeFunction").getText();
            } catch (Exception e) {
                LOGGER.log(Level.INFO, "Schaftholz ist: {0}", actual.stemVolumeFunctionXML);
            }
        }
        if (actual.coarseRootBiomass.trim().length() < 1) {
            actual.coarseRootBiomass = with.getChild("CoarseRootBiomass").getText();
        }
        if (actual.smallRootBiomass.trim().length() < 1) {
            actual.smallRootBiomass = with.getChild("SmallRootBiomass").getText();
        }
        if (actual.fineRootBiomass.trim().length() < 1) {
            actual.fineRootBiomass = with.getChild("FineRootBiomass").getText();
        }
        if (actual.totalRootBiomass.trim().length() < 1) {
            actual.totalRootBiomass = with.getChild("TotalRootBiomass").getText();
        }
    }

    private void define(int code, SpeciesDef actual, Element def, int hlc) {
        actual.code = code;
        actual.handledLikeCode = hlc;
        actual.shortName = def.getChild("ShortName").getText();
        actual.longName = def.getChild("LongName").getText();
        actual.latinName = def.getChild("LatinName").getText();
        actual.internalCode = Integer.parseInt(def.getChild("InternalCode").getText());
        actual.codeGroup = Integer.parseInt(def.getChild("CodeGroup").getText());
        actual.heightCurve = Integer.parseInt(def.getChild("HeightCurve").getText());
        actual.crownType = Integer.parseInt(def.getChild("CrownType").getText());
        actual.heightIncrementError = Double.parseDouble(def.getChild("HeightIncrementError").getText());
        actual.diameterIncrementError = Double.parseDouble(def.getChild("DiameterIncrementError").getText());
        actual.maximumAge = Integer.parseInt(def.getChild("MaximumAge").getText());
        actual.ingrowthXML = def.getChild("Ingrowth").getText();
        actual.targetDiameter = Double.parseDouble(def.getChild("TargetDiameter").getText());
        actual.cropTreeNumber = stripCommentsFromInt(def.getChild("CropTreeNumber").getText(), -9);
        actual.heightOfThinningStart = Double.parseDouble(def.getChild("HeightOfThinningStart").getText());
        actual.moderateThinning = def.getChild("ModerateThinning").getText();
        actual.colorXML = def.getChild("Color").getText();
        actual.competitionXML = def.getChild("Competition").getText();
        actual.taperFunctionXML = def.getChild("TaperFunction").getText();
        try {
            actual.stemVolumeFunctionXML = def.getChild("StemVolumeFunction").getText();
        } catch (Exception e) {
            LOGGER.log(Level.INFO, "Schaftholz ist: {0}", actual.stemVolumeFunctionXML);
        }
        actual.coarseRootBiomass = def.getChild("CoarseRootBiomass").getText();
        actual.smallRootBiomass = def.getChild("SmallRootBiomass").getText();
        actual.fineRootBiomass = def.getChild("FineRootBiomass").getText();
        actual.totalRootBiomass = def.getChild("TotalRootBiomass").getText();
        //TGFunctions
        actual.uniformHeightCurveXML = initTGFunction(def.getChild("UniformHeightCurveXML").getText().trim());
        actual.heightVariationXML = initTGFunction(def.getChild("HeightVariation").getText().trim());
        actual.diameterDistributionXML = initTGFunction(def.getChild("DiameterDistributionXML").getText().trim());
        actual.volumeFunctionXML = initTGFunction(def.getChild("VolumeFunctionXML").getText().trim());
        actual.crownwidthXML = initTGFunction(def.getChild("Crownwidth").getText().trim());
        actual.crownbaseXML = initTGFunction(def.getChild("Crownbase").getText().trim());
        actual.siteindexXML = initTGFunction(def.getChild("SiteIndex").getText().trim());
        actual.siteindexHeightXML = initTGFunction(def.getChild("SiteIndexHeight").getText().trim());
        actual.potentialHeightIncrementXML = initTGFunction(def.getChild("PotentialHeightIncrement").getText().trim());
        actual.heightIncrementXML = initTGFunction(def.getChild("HeightIncrement").getText().trim());
        actual.diameterIncrementXML = initTGFunction(def.getChild("DiameterIncrement").getText().trim());
        actual.maximumDensityXML = initTGFunction(def.getChild("MaximumDensity").getText().trim());
        actual.decayXML = initTGFunction(def.getChild("Decay").getText().trim());
    }
    */

    private int stripCommentsFromInt(String orig, int stdValue) {
        if (orig == null || orig.equals("")) {
            return stdValue;
        }
        return Integer.parseInt(orig.split("[/][*].+?[*][/]")[0].trim());
    }

    public TGFunction initTGFunction(String xmlText) {
        if (xmlText == null) {
            return new TGTextFunction();
        }
        if (xmlText.length() == 0) {
            return new TGTextFunction();
        }
        if (xmlText.startsWith("CLASS:")) {
            TGClassFunction f = new TGClassFunction();
            f.init(xmlText);
            return f;
        } else {
            TGTextFunction f = new TGTextFunction();
            f.init(xmlText);
            return f;
        }
    }

    public boolean isLoaded() {
        return loaded;
    }

    public URL getActualURL() {
        return actualurl;
    }

    public int getSize() {
        return spcdef.size();
    }

    public int[] getSpeciesCodes() {
        if (loaded) {
            int[] list = new int[spcdef.size()];
            Iterator<Integer> it = spcdef.keySet().iterator();
            int index = 0;
            while (it.hasNext()) {
                list[index] = it.next();
                index++;
            }
            return list;
        } else {
            return null;
        }
    }

    public SpeciesDef getByCode(int code) {       
        return spcdef.get(code);
    }

    /**
     * insert a new species only if the map is loaded and the map does not
     * contain a species with code code returns the new and empty SpeciesDef
     * object or null no new species is inserted
     *
     * @param code the species code
     * @return species definition for speecies with defined code
     */
    public SpeciesDef insertSpecies(int code) {
        if (loaded && getByCode(code) == null) {
            SpeciesDef spec = new SpeciesDef();
            spcdef.put(code, spec);
            return spec;
        }
        return null;
    }

    public void removeSpecies(int code) {
        if (loaded) {
            spcdef.remove(code);
        }
    }

    @Override
    public String toString() {
        return "SpeciedDefMap [size: " + getSize() + "; URL:" + getActualURL().toString() + "]";
    }

    /* writes species information for all species of one stand to a html file in
     * specified path with specified filename
     * and returns the complete cannonical path of the output file.
     */
    public String listAllSpeciesDefinition(Stand st, String path, String fname) {
        File file = new File(path, fname);
        String filename;
        try {
            filename = file.getCanonicalPath();
        } catch (IOException ex) {
            LOGGER.log(Level.SEVERE, "treegross", ex);
            return null;
        }
        try (PrintWriter out = new PrintWriter(new OutputStreamWriter(new FileOutputStream(filename)))) {
            out.println("<html><head><title>Simulator Species Definition</title></head><body>");
            out.println("<h2 style=\"text-align:center;\">Simulator Species Definition</h2> ");
            for (int i = 0; i < st.nspecies; i++) {
                out.println("<p>");
                int m = -9;
                if (st.sp[i].spDef.latinName.contains("http")) {
                    m = st.sp[i].spDef.latinName.indexOf("http") - 1;
                }
                String txt = st.sp[i].spDef.latinName;
                if (m > 1) {
                    txt = "<a href=" + st.sp[i].spDef.latinName.substring(m + 1, st.sp[i].spDef.latinName.length()) + ">" + st.sp[i].spDef.latinName.substring(0, m) + "</a>";
                }
                out.println("</p><p><b>Baumart: " + st.sp[i].code + " " + st.sp[i].spDef.longName + "  " + txt + "</b>");
                out.println("<br>   Kronenbreite [m] = " + st.sp[i].spDef.crownwidthXML);
                out.println("<br>   Kronenansatz [m] = " + st.sp[i].spDef.crownbaseXML);
                out.println("<br>   Bonitöt      [m] = " + st.sp[i].spDef.siteindexXML);
                out.println("<br>   Potentielle Höhenzuwachs [%] = " + st.sp[i].spDef.potentialHeightIncrementXML);
                out.println("<br>   Höhenzuwachsmodulation [%] = " + st.sp[i].spDef.heightIncrementXML);
                out.println("<br>   Standardabweichung Höhenzuwachs [m] = " + st.sp[i].spDef.heightIncrementError);
                out.println("<br>   Grundflächenzuwachs [cm²] = " + st.sp[i].spDef.diameterIncrementXML);
                out.println("<br>   Standardabweichung Grundflächenzuwachs [m²] = " + st.sp[i].spDef.diameterIncrementError);
                out.println("<br>   Baumindividueller Error (Slope, Intercept, General) = " + st.sp[i].spDef.diameterTreeErrorXML);
                out.println("<br>   Maximale Dichte [m²/ha] = " + st.sp[i].spDef.maximumDensityXML);
                out.println("<br>   Volumenfunktion [m³] = " + st.sp[i].spDef.volumeFunctionXML);
                out.println("<br>   Durchmesserverteilung : " + st.sp[i].spDef.diameterDistributionXML);
                out.println("<br>   Höhenkurvenfunktion = " + st.sp[i].spDef.heightCurve);
                out.println("<br>   Einheitshöhenkurve [m] = " + st.sp[i].spDef.uniformHeightCurveXML);
                out.println("<br>   Höhenkurvenvariation [m] = " + st.sp[i].spDef.heightVariationXML);
                out.println("<br>   Totholzzerfall [%] = " + st.sp[i].spDef.decayXML);
                out.println("<br>   Kronendarstellung = " + st.sp[i].spDef.crownType);
                out.println("<br>   Baumartenfarbe [RGB] = " + st.sp[i].spDef.colorXML);
                out.println("</p>");
            }
            //out.println("</TABLE>");
            out.println("<br>created by TreeGroSS (" + st.modelRegion + ")</br></body></html>");
        } catch (FileNotFoundException ex) {
            LOGGER.log(Level.SEVERE, "treegross", ex);
            return null;
        }
        return filename;
    }

    /* writes species information for active species of one stand to a html file in
     * specified path with specified filename
     * and returns the complete cannonical path of the output file.
     */
    public String listCurrentSpeciesDefinition(Stand st, String path, String fname) throws IOException {
        File file = new File(path, fname);
        String filename = file.getCanonicalPath();
        try (PrintWriter out = new PrintWriter(new OutputStreamWriter(new FileOutputStream(filename)))) {
            out.println("<html><head><title>Simulator Species Definition</title></head><body>");
            out.println("<h2 style=\"text-align:center;\">Simulator Species Definition</h2> ");
            if (st.nspecies > 0 && st.ingrowthActive) {
                try {
                    String modelPlugIn =/*"treegross.base."+*/ st.sp[0].spDef.ingrowthXML;
                    PlugInIngrowth ig = (PlugInIngrowth) Class.forName(modelPlugIn).getDeclaredConstructor().newInstance();
                    out.println("<p><b>Aktivieres Einwuchsmodell: " + ig.getModelName() + "</b></p>");
                } catch (Exception e) {
                    LOGGER.log(Level.SEVERE, "ERROR in Class Ingrowth2", e);
                }
            }
            for (int i = 0; i < st.nspecies; i++) {
                out.println("<p>");
                int m = -9;
                if (st.sp[i].spDef.latinName.contains("http")) {
                    m = st.sp[i].spDef.latinName.indexOf("http") - 1;
                }
                String txt = st.sp[i].spDef.latinName;
                if (m > 1) {
                    txt = "<a href=" + st.sp[i].spDef.latinName.substring(m + 1, st.sp[i].spDef.latinName.length()) + ">" + st.sp[i].spDef.latinName.substring(0, m) + "</a>";
                }
                out.println("</p>");
                out.println("<p><b>Baumart: " + st.sp[i].code + " " + st.sp[i].spDef.longName + "  " + txt + "</b>");
                out.println("<br>   Kronenbreite [m] = " + st.sp[i].spDef.crownwidthXML);
                out.println("<br>   Kronenansatz [m] = " + st.sp[i].spDef.crownbaseXML);
                out.println("<br>   Bonität      [m] = " + st.sp[i].spDef.siteindexXML);
                out.println("<br>   Potentielle Höhenzuwachs [%] = " + st.sp[i].spDef.potentialHeightIncrementXML);
                out.println("<br>   Höhenzuwachsmodulation [%] = " + st.sp[i].spDef.heightIncrementXML);
                out.println("<br>   Standardabweichung Höhenzuwachs [m] = " + st.sp[i].spDef.heightIncrementError);
                out.println("<br>   Grundflächenzuwachs [cm²] = " + st.sp[i].spDef.diameterIncrementXML);
                out.println("<br>   Standardabweichung Grundflächenzuwachs [m²] = " + st.sp[i].spDef.diameterIncrementError);
                out.println("<br>   Baumindividueller Error (Slope, Intercept, General) = " + st.sp[i].spDef.diameterTreeErrorXML);
                out.println("<br>   Maximale Dichte [m²/ha] = " + st.sp[i].spDef.maximumDensityXML);
                out.println("<br>   Volumenfunktion [m³] = " + st.sp[i].spDef.volumeFunctionXML);
                out.println("<br>   Durchmesserverteilung : " + st.sp[i].spDef.diameterDistributionXML);
                out.println("<br>   Höhenkurvenfunktion = " + st.sp[i].spDef.heightCurve);
                out.println("<br>   Einheitshöhenkurve [m] = " + st.sp[i].spDef.uniformHeightCurveXML);
                out.println("<br>   Höhenkurvenvariation [m] = " + st.sp[i].spDef.heightVariationXML);
                out.println("<br>   Totholzzerfall [%] = " + st.sp[i].spDef.decayXML);
                out.println("<br>   Kronendarstellung = " + st.sp[i].spDef.crownType);
                out.println("<br>   Baumartenfarbe [RGB] = " + st.sp[i].spDef.colorXML);
                out.println("</p>");
            }
            out.println("<br> created by TreeGrOSS (" + st.modelRegion + ")</br></body></html>");
        }
        return filename;
    }

    public String listSpeciesCode(int code, String path, String fname2) {
        File file = new File(path, fname2);
        String filename;
        try {
            filename = file.getCanonicalPath();
        } catch (IOException ex) {
            LOGGER.log(Level.SEVERE, "treegross", ex);
            return null;
        }
        try (PrintWriter out = new PrintWriter(new OutputStreamWriter(new FileOutputStream(filename)))) {
            out.println("<html><head><title>Simulator Species Definition</title></head><body>");
            out.println("<h2 style=\"text-align:center;\">" + "Species Code" + "</h2><p> ");
            SpeciesDef sd = this.getByCode(code);
            int mm = -9;
            if (sd.latinName.contains("http")) {
                mm = sd.latinName.indexOf("http") - 1;
            }
            String txt = sd.latinName;
            if (mm > 1) {
                txt = "<a href=" + sd.latinName.substring(mm + 1, sd.latinName.length()) + ">" + sd.latinName.substring(0, mm) + "</a>";
            }
            out.println("<br>Baumart: " + sd.code + " " + sd.shortName + " " + sd.longName + " " + txt);
            out.println("<br><hr>created by TreeGrOSS</body></html>");
        } catch (FileNotFoundException ex) {
            LOGGER.log(Level.SEVERE, "treegross", ex);
        }
        return filename;
    }
}
